'use strict';

var test = require('tape');
var inspect = require('object-inspect');
var forEach = require('foreach');

var SLOT = require('../');

test('assert', function (t) {
	forEach([null, undefined, true, false, 'foo', '', 42, 0], function (primitive) {
		t['throws'](
			function () { SLOT.assert(primitive, ''); },
			TypeError,
			inspect(primitive) + ' is not an Object'
		);
	});

	forEach([null, undefined, true, false, 42, 0, {}, [], function () {}, /a/g], function (nonString) {
		t['throws'](
			function () { SLOT.assert({}, nonString); },
			TypeError,
			inspect(nonString) + ' is not a String'
		);
	});

	t['throws'](
		function () { SLOT.assert({}, 'whatever'); },
		TypeError,
		'nonexistent slot throws'
	);

	var o = {};
	SLOT.set(o, 'x');
	t.doesNotThrow(function () { SLOT.assert(o, 'x'); }, 'existent slot noops');

	t.end();
});

test('has', function (t) {
	forEach([null, undefined, true, false, 'foo', '', 42, 0], function (primitive) {
		t['throws'](
			function () { SLOT.has(primitive, ''); },
			TypeError,
			inspect(primitive) + ' is not an Object'
		);
	});

	forEach([null, undefined, true, false, 42, 0, {}, [], function () {}, /a/g], function (nonString) {
		t['throws'](
			function () { SLOT.has({}, nonString); },
			TypeError,
			inspect(nonString) + ' is not a String'
		);
	});

	var o = {};

	t.equal(SLOT.has(o, 'nonexistent'), false, 'nonexistent slot yields false');

	SLOT.set(o, 'foo');
	t.equal(SLOT.has(o, 'foo'), true, 'existent slot yields true');

	t.end();
});

test('get', function (t) {
	forEach([null, undefined, true, false, 'foo', '', 42, 0], function (primitive) {
		t['throws'](
			function () { SLOT.get(primitive, ''); },
			TypeError,
			inspect(primitive) + ' is not an Object'
		);
	});

	forEach([null, undefined, true, false, 42, 0, {}, [], function () {}, /a/g], function (nonString) {
		t['throws'](
			function () { SLOT.get({}, nonString); },
			TypeError,
			inspect(nonString) + ' is not a String'
		);
	});

	var o = {};
	t.equal(SLOT.get(o, 'nonexistent'), undefined, 'nonexistent slot is undefined');

	var v = {};
	SLOT.set(o, 'f', v);
	t.equal(SLOT.get(o, 'f'), v, '"get" retrieves value set by "set"');

	t.end();
});

test('set', function (t) {
	forEach([null, undefined, true, false, 'foo', '', 42, 0], function (primitive) {
		t['throws'](
			function () { SLOT.set(primitive, ''); },
			TypeError,
			inspect(primitive) + ' is not an Object'
		);
	});

	forEach([null, undefined, true, false, 42, 0, {}, [], function () {}, /a/g], function (nonString) {
		t['throws'](
			function () { SLOT.set({}, nonString); },
			TypeError,
			inspect(nonString) + ' is not a String'
		);
	});

	var o = function () {};
	t.equal(SLOT.get(o, 'f'), undefined, 'slot not set');

	SLOT.set(o, 'f', 42);
	t.equal(SLOT.get(o, 'f'), 42, 'slot was set');

	SLOT.set(o, 'f', Infinity);
	t.equal(SLOT.get(o, 'f'), Infinity, 'slot was set again');

	t.end();
});
;